/* * I2C client/driver for the Maxim/Dallas DS2782 Stand-Alone Fuel Gauge IC
 *
 * Copyright (C) 2009 Bluewater Systems Ltd
 *
 * Author: Ryan Mallon <ryan@bluewatersys.com>
 *
 *
 * DS2786 added by Yulia Vilensky <vilensky@compulab.co.il>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/device.h>
#include <linux/mutex.h>
#include <linux/errno.h>
#include <linux/swab.h>
#include <linux/workqueue.h>
#include <linux/i2c.h>
#include <linux/idr.h>
#include <linux/power_supply.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/pmic_external.h>
#include <linux/pmic_status.h>
#include <linux/mfd/da9052/reg.h>
#include <linux/platform_device.h>
#include <linux/ds2782_battery.h>

#include <linux/crestron.h>

#include <mach/gpio.h>

#include <linux/proc_fs.h>  /* Necessary because we use proc fs */
#include <asm/uaccess.h>    /* for copy_*_user */

#define BOGUS_VALUE 99  //RJK need to implement section with this value
#define BOGUS_VALUE_CAP 0x87803  //RJK need to implement section with this value

#define POWER_PROCFS_DIRECTORY 		"power"  // /proc/power is location to read/write battery updates
#define POWER_PROCFS_CMD_FILENAME 	"batCmd"  // /proc/power is location to read/write battery updates
#define POWER_PROCFS_EXTCMD_FILENAME 	"batExtCmd"  // /proc/power is location to read/write battery updates
#define PROCFS_MAX_SIZE     2048

/** * The structure keeping information about the /proc file */
static struct proc_dir_entry *PowerDir;
static struct proc_dir_entry *CmdFile;
static struct proc_dir_entry *ExtCmdFile;

/** * The buffer (2k) for this module */
static char gPower_buffer[PROCFS_MAX_SIZE];

/** * The size of the data held in the buffer */
static unsigned long gPower_buffer_size = 0;

/* power Pin, used to check if on AC or DC */
#define EXT_PWR_N                       (1*32 + 30)     /* GPIO_2_30 */
#define DOCK_CHG_INDICATOR              (6*32 + 10)     /* GPIO_7_10 */
#define GG_PROG_PULSE                   (2*32 + 25)     /* GPIO_3_25 */

/* EEPROM Block */
#define DS2782_REG_RSNSP	0x69	/* Sense resistor value */

/* Current unit measurement in uA for a 1 milli-ohm sense resistor */
#define CHG_CURRENT(x) (x/30)

struct ds278x_info;

struct ds278x_battery_ops {
        int (*get_battery_current)(struct ds278x_info *info, int *current_uA);
        int (*get_battery_voltage)(struct ds278x_info *info, int *voltage_uA);
        int (*get_battery_capacity)(struct ds278x_info *info, u8 *capacity_uA);
        int (*get_battery_temp)(struct ds278x_info *info, int *bat_temp);
};

static int ds2786_battery_read_status(struct ds278x_info *info);

/* exposed functions for battery monitoring using ds2786
 */
u8 get_ds2786_capacity(void);
u16 get_ds2786_voltage(void);
int get_ds2786_current(void);

#define to_ds278x_info(x) container_of(x, struct ds278x_info, battery)

struct ds278x_info {

	struct i2c_client	*client;

        struct delayed_work monitor_work;
	struct power_supply	ac;
	struct power_supply	battery;

	struct ds278x_battery_ops  *ops;
	int			id;
	int                     rsns;

	struct device *dev;

	int power_led_enable;   // 0=disabled; 1=enabled
        int battery_critical_level_thresh;      // 0-100%
        int battery_warning_level_thresh;       // 0-100%
        int warning_led_flash_rate;
        int critical_led_flash_rate;
        int charging_led_flash_rate;
        int current_cpu_temperature;
       	int current_battery_temperature;

        int voltage_raw;
        int voltage_uV;
        int current_raw;
        int current_uA;
        int rated_capacity;             /* units of µAh */
        u8 rem_capacity;               /* percentage */
        int full_active_uAh;            /* units of µAh */
        int empty_uAh;                  /* units of µAh */
        int life_sec;                   /* units of seconds */
        int charge_status;              /* POWER_SUPPLY_STATUS_* */

        int full_counter;       // 0= battery not full
        int charger_voltage_uV;
        int accum_current_uAh;

        int on_ac;   // if !on_ac; charger is online
        int battery_level;
        int battery_status;     //charging or not charging
        struct system_power_status_ex2 extCmd;


//	struct workqueue_struct *monitor_wqueue;

};

struct led_control {
	struct delayed_work led_work;
	int interval;
};

struct ds278x_info *g_control;

int g_led_state=1;

/* function prototype */
static void enableLed(int st);
static void blinkLed(void);

static inline int ds278x_read_reg16(struct ds278x_info *info, int reg_msb, s16 *val);
static inline int ds278x_read_reg(struct ds278x_info *info, int reg, u8 *val);


static DEFINE_IDR(battery_id);
static DEFINE_MUTEX(battery_lock);

/* charger functions */
static int enableBattCharger()
{
	// pmic_write_reg(DA9052_CHGBUCK_REG, 0x9F, 0xFF);
	 enableLed(1);
}
static int disableBattCharger()
{
	 //pmic_write_reg(DA9052_CHGBUCK_REG, 0x9F, 0xFF);
	 enableLed(0);
}


// RJK: Crestron
void UpdateGasGaugeEE(struct ds278x_info *info, DS2786B_GAS_GAUGE_CFG_BLOCK* gasGaugeUpdateBlock)
{
	// Steps needed for programming the EEPROM on the gas gauge
    	//
    	// Step 1 - Write all the registers needs to be programmed
	// Step 2 - Apply programming voltage to the chip.
	// Step 3 - Set the COPY bit in the Command register 
	// Step 4 - Wait for 12ms (min 3.1 ms, max 14ms)
	// Step 5 - Reset tje COPY bit in the command register
	// Step 6 - Remove the programming voltage.

	u32 dwNoOfRegisters, dwBytesToWrite, baseTckCnt, i;
    	u32 *pDwPtr;
    	u8 address, data8;
    	u16 data16;
	u16 tempval;

	pDwPtr = (u32*)gasGaugeUpdateBlock;

	dwNoOfRegisters = sizeof (DS2786B_GAS_GAUGE_CFG_BLOCK) / sizeof (u32);
	printk(KERN_ERR "IOCTL_UPDATE_GAS_GAUGE: No of reg to Program: %d\r\n", dwNoOfRegisters);

	// Now write all the registers to be programmed
	for (i = 0; i < dwNoOfRegisters; i++)
	{
		// Read the no of bytes to write in the word
		dwBytesToWrite =  ((*pDwPtr &  0x00FF0000) >> 16);
		// See if it is a non-zero item
		if (dwBytesToWrite)
		{
			// Get the address byte in the word
			address = (u8) ((*pDwPtr & 0xFF000000) >> 24);
			if (dwBytesToWrite == 2)    // 2 bytes
			{
				data16 = (u16)(*pDwPtr & 0x0000FFFF);
				i2c_smbus_write_word_data(info->client, address, data16);
				//tempval=i2c_smbus_read_word_data(info->client, address);
				printk(KERN_ERR "write Reg16: 0x%02x, Val: 0x%04x\r\n", address, data16);
				//printk(KERN_ERR "read Reg16: 0x%02x, Val: 0x%04x\r\n", address, tempval);
			}
			else                        // 1 byte
			{
				data8 = (u8)(*pDwPtr & 0x0000FF);
				i2c_smbus_write_byte_data(info->client, address, data8);
				//tempval=i2c_smbus_read_byte_data(info->client, address);
				printk(KERN_ERR "write Reg16: 0x%02x, Val: 0x%04x\r\n", address, data8);
				//printk(KERN_ERR "read Reg16: 0x%02x, Val: 0x%04x\r\n", address, tempval);
			}
		}
		else
		{
			// Throw a message for which registers are not programmed
			printk(KERN_ERR "Update Gas gauge: Register [0x%02x] not programmed. Uses default.\r\n",
							   (u8) ((*pDwPtr & 0xFF000000) >> 24));
		}
		pDwPtr++;
	}
	// Apply 14V pulse for 12 milliseconds
	gpio_set_value(GG_PROG_PULSE, 0);
	gpio_set_value(GG_PROG_PULSE, 1);

	// Wait for a 2 ms for the voltage to stabilize (just for safety)
	msleep(2);


	// Set COPY bit in the command register.
	data8=i2c_smbus_read_byte_data(info->client, DS2786B_COMMAND);
	data8 |= COMMAND_BIT_COPY;
	i2c_smbus_write_byte_data(info->client, DS2786B_COMMAND, data8);

	// Wait for 12 ms
	msleep(12);

	// clear the COPY  bit
	data8=i2c_smbus_read_byte_data(info->client, DS2786B_COMMAND);
	data8 &= ~COMMAND_BIT_COPY;
	i2c_smbus_write_byte_data(info->client, DS2786B_COMMAND, data8);

	// Remove the 14 V pulse.
	gpio_set_value(GG_PROG_PULSE, 0);

	// Finally start the SOCV calculation. Bit gets cleared automatically
	// No need to do read-modify write!
	data8 = COMMAND_BIT_SOCV;
	i2c_smbus_write_byte_data(info->client, DS2786B_COMMAND, data8);

}

void initDs2782ExtCmds(void) {

    g_control->extCmd.ACLineStatus         	= AC_LINE_OFFLINE;
    g_control->extCmd.BatteryFlag         	= BATTERY_FLAG_UNKNOWN;
    g_control->extCmd.BatteryLifePercent         = 0;
    g_control->extCmd.Reserved1                  = 0;
    g_control->extCmd.BatteryLifeTime            = BATTERY_LIFE_UNKNOWN;
    g_control->extCmd.BatteryFullLifeTime        = BATTERY_LIFE_UNKNOWN;

    // NOTE:
    //  If backup battery exists these fields should be filled
    // with correct information.  For now just indicate full charge
    //    
    g_control->extCmd.Reserved2                  = 0;
    g_control->extCmd.Reserved3                  = 0;    
    g_control->extCmd.BackupBatteryLifeTime      = BATTERY_LIFE_UNKNOWN;
    g_control->extCmd.BackupBatteryFullLifeTime  = BATTERY_LIFE_UNKNOWN;
    g_control->extCmd.BackupBatteryFlag          = BATTERY_FLAG_UNKNOWN;
    g_control->extCmd.BackupBatteryLifePercent   = 0 ;    


    g_control->extCmd.BatteryChemistry           = BATTERY_CHEMISTRY_LION;
    g_control->extCmd.BatteryVoltage             = 0; 
    g_control->extCmd.BatteryCurrent             = 500;
    g_control->extCmd.BatteryAverageCurrent      = 500;
    g_control->extCmd.BatteryAverageInterval     = 5000;
    g_control->extCmd.BatterymAHourConsumed      = BATTERY_LIFE_UNKNOWN;
   // g_control->extCmd.BatteryTemperature         = 30;
    g_control->extCmd.BackupBatteryVoltage       = 3000;

}

void GasGaugeInitConfReg( struct ds278x_info *info ) 
{
	u8 val;
	int err;

printk(KERN_ERR " RJK:: GasGaugeInitConfReg\n");
	mutex_lock(&battery_lock);
        err = ds278x_read_reg(info, DS2786B_STATUSCONF, &val);
	mutex_unlock(&battery_lock);
        if (err)
	{
                return err;
	}
	else
	{		
		if (val & STATUS_CONFIG_BIT_PORF)
		{
			printk(KERN_ERR "GAS Gauge is 1st time powered up, clearing PORF bit\n");
			val = val & ~STATUS_CONFIG_BIT_PORF;

			i2c_smbus_write_byte_data(info->client, DS2786B_STATUSCONF, val);

	                // Set the SOCV bit to start SOCV calculation
        	        // Not necessary to do a read modify write, cause all are the command
           	        // bits and they are all 0 by default. After setting, they clear automatically.
        	        val = COMMAND_BIT_SOCV;

			i2c_smbus_write_byte_data(info->client, DS2786B_COMMAND, val);
		}
	}
}

void GasGaugeInitEE(struct ds278x_info *info)
{
	DS2786B_GAS_GAUGE_CFG_BLOCK gasGaugeUpdateBlock;

	gasGaugeUpdateBlock.ds2786b_ee_cur_offset_bias_reg = DS2786B_CUR_OFFSET_BIAS_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_1_reg = DS2786B_CAPACITY_1_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_2_reg = DS2786B_CAPACITY_2_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_3_reg = DS2786B_CAPACITY_3_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_4_reg = DS2786B_CAPACITY_4_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_5_reg = DS2786B_CAPACITY_5_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_6_reg = DS2786B_CAPACITY_6_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_capacity_7_reg = DS2786B_CAPACITY_7_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_0_reg = DS2786B_VOLTG_BRK_PT_0_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_1_reg = DS2786B_VOLTG_BRK_PT_1_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_2_reg = DS2786B_VOLTG_BRK_PT_2_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_3_reg = DS2786B_VOLTG_BRK_PT_3_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_4_reg = DS2786B_VOLTG_BRK_PT_4_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_5_reg = DS2786B_VOLTG_BRK_PT_5_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_6_reg = DS2786B_VOLTG_BRK_PT_6_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_7_reg = DS2786B_VOLTG_BRK_PT_7_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_8_reg = DS2786B_VOLTG_BRK_PT_8_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_voltg_brk_pt_8_reg = DS2786B_VOLTG_BRK_PT_8_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_init_cap_scaling_reg = DS2786B_INIT_CAP_SCALING_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_ocv_cur_thr_reg = DS2786B_OCV_CUR_THR_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_ocv_dvdt_thr_reg = DS2786B_OCV_DVDT_THR_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_i2c_addr_conf_reg = DS2786B_I2C_ADDR_CONF_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_learn_thr_reg = DS2786B_LEARN_THR_VAL;
	gasGaugeUpdateBlock.ds2786b_ee_user_eeprom_reg = DS2786B_USER_EEPROM_VAL;

	UpdateGasGaugeEE(info, &gasGaugeUpdateBlock);
}

/* Procfs functions */
/** * This funtion is called when the /proc power file is read */
static ssize_t power_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
  	char * p = gPower_buffer;
  	static int finished=0;

  	/* needed to stop from continuously printing */
  	if ( finished == 1 ) { finished=0; return 0; }
  	finished = 1;
   	p += sprintf ( p, "ON_AC=%04d\n" , g_control->on_ac );
   	p += sprintf ( p, "BATTERY_LEVEL=%04d\n" , g_control->battery_level );
   	p += sprintf ( p, "BATTERY_CHARGING=%04d\n" , g_control->battery_status );
   	p += sprintf ( p, "POWER_LED_ENABLE=%04d\n" , g_control->power_led_enable );
   	p += sprintf ( p, "BATTERY_CRITICAL_LEVEL_THRESH=%04d\n" , g_control->battery_critical_level_thresh );
   	p += sprintf ( p, "BATTERY_WARNING_LEVEL_THRESH=%04d\n" , g_control->battery_warning_level_thresh );
   	p += sprintf ( p, "WARNING_LED_FLASH_RATE=%04d\n" , g_control->warning_led_flash_rate );
   	p += sprintf ( p, "CRITICAL_LED_FLASH_RATE=%04d\n" , g_control->critical_led_flash_rate );
   	p += sprintf ( p, "CHARGING_LED_FLASH_RATE=%04d\n" , g_control->charging_led_flash_rate );
   	p += sprintf ( p, "CURRENT_CPU_TEMPERATURE=%04d\n" , g_control->current_cpu_temperature );
   	p += sprintf ( p, "CURRENT_BATTERY_TEMPERATURE=%04d\n" , g_control->current_battery_temperature );

   	gPower_buffer_size = p-gPower_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gPower_buffer, gPower_buffer_size) ) {
                return -EFAULT;
        }

  	return gPower_buffer_size;
}

/** * This funtion is called when the /proc power file is read */
static ssize_t power_ext_read(struct file *filp,  /* see include/linux/fs.h   */
                             char *buffer,      /* buffer to fill with data */
                             size_t length,     /* length of the buffer     */
                             loff_t * offset)
{
        char * p = gPower_buffer;
        static int finished=0;

        /* needed to stop from continuously printing */
        if ( finished == 1 ) { finished=0; return 0; }
        finished = 1;
        p += sprintf ( p, "EXTGETCMD_BATTERYCHEMISTRY=%04d\n" , g_control->extCmd.BatteryChemistry );
        p += sprintf ( p, "EXTGETCMD_CHARGEMODE=%04d\n" , g_control->extCmd.BatteryFlag );
        p += sprintf ( p, "EXTGETCMD_CCCURREENT=%04d\n" , g_control->extCmd.BatteryCurrent );
        p += sprintf ( p, "EXTGETCMD_CVVOLTAGE=%04d\n" , g_control->extCmd.BatteryVoltage );
        p += sprintf ( p, "EXTGETCMD_AUTOCHARGE=%04d\n" , 0x0001);  //forced to 1

	//RJK need to implement this........
        p += sprintf ( p, "EXTGETCMD_TOTALCHARGETIME=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_CURRENTCHARGETIME=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_LASTWAKEUP=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_SUSPENDCHARGE=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_CHARGECURRENT=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_FULLCAPACITY=%04d\n" , BOGUS_VALUE);
        p += sprintf ( p, "EXTGETCMD_MAXCOMMANDS=%04d\n" , BOGUS_VALUE);
	
	//RJK  this will be moved to PM, but using it here for development only
        p += sprintf ( p, "CAPABILITIES=0x%08d\n" , BOGUS_VALUE_CAP);

        gPower_buffer_size = p-gPower_buffer;

        /*
         * We use put_to_user to copy the string from the kernel's
         * memory segment to the memory segment of the process
         * that called us. get_from_user, BTW, is
         * used for the reverse.
         */
        if ( copy_to_user(buffer, gPower_buffer, gPower_buffer_size) ) {
                return -EFAULT;
        }

        return gPower_buffer_size;
}

static ssize_t power_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
	char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;

        gPower_buffer_size = len;
        if (gPower_buffer_size > PROCFS_MAX_SIZE ) {
                gPower_buffer_size = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(gPower_buffer, buffer, gPower_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "POWER_LED_ENABLE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->power_led_enable );
	  enableLed(g_control->power_led_enable);	
        }
        if ( strcmp ( tag, "BATTERY_CRITICAL_LEVEL_THRESH" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->battery_critical_level_thresh );
        }
        if ( strcmp ( tag, "BATTERY_WARNING_LEVEL_THRESH" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->battery_warning_level_thresh );
        }
        if ( strcmp ( tag, "WARNING_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->warning_led_flash_rate );
        }
        if ( strcmp ( tag, "CRITICAL_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->critical_led_flash_rate );
        }
        if ( strcmp ( tag, "CHARGING_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->charging_led_flash_rate );
        }
        return gPower_buffer_size;

}

static ssize_t power_ext_write(struct file *file, const char *buffer, size_t len, loff_t * off)
{
        char* tag = NULL;
        char* value = NULL;
        char** tempPtr = &buffer;

        gPower_buffer_size = len;
        if (gPower_buffer_size > PROCFS_MAX_SIZE ) {
                gPower_buffer_size = PROCFS_MAX_SIZE;
        }

        if ( copy_from_user(gPower_buffer, buffer, gPower_buffer_size) )
        {
                return -EFAULT;
        }
        tag = strsep ( tempPtr, "=" );

        if ( strcmp ( tag, "POWER_LED_ENABLE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->power_led_enable );
          enableLed(g_control->power_led_enable);
        }
        if ( strcmp ( tag, "BATTERY_CRITICAL_LEVEL_THRESH" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->battery_critical_level_thresh );
        }
        if ( strcmp ( tag, "BATTERY_WARNING_LEVEL_THRESH" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->battery_warning_level_thresh );
        }
        if ( strcmp ( tag, "WARNING_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->warning_led_flash_rate );
        }
        if ( strcmp ( tag, "CRITICAL_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->critical_led_flash_rate );
        }
        if ( strcmp ( tag, "CHARGING_LED_FLASH_RATE" ) == 0 )
        {
          value = strsep ( tempPtr, "=" );
          sscanf ( value, "%d", &g_control->charging_led_flash_rate );
        }
        return gPower_buffer_size;

}

static int module_permission(struct inode *inode, int op, struct nameidata *foo)
{
//  if ( op == 2 ) // no writes
//  {
//    return -EACCES;
//  }

  return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int power_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is opened - we don't really care about
 * that, but it does mean we need to increment the
 * module's reference count.
 */
int power_ext_open_procfs(struct inode *inode, struct file *file)
{
        try_module_get(THIS_MODULE);
        return 0;
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int power_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

/*
 * The file is closed - again, interesting only because
 * of the reference count.
 */
int power_ext_close_procfs(struct inode *inode, struct file *file)
{
        module_put(THIS_MODULE);
        return 0;               /* success */
}

static struct file_operations File_Ops_Power_File = {
        .read    = power_read,
        .write   = power_write,
        .open    = power_open_procfs,
        .release = power_close_procfs,
};

static struct file_operations File_Ops_ExtPower_File = {
        .read    = power_ext_read,
        .write   = power_ext_write,
        .open    = power_ext_open_procfs,
        .release = power_ext_close_procfs,
};

/*
 * Inode operations for our proc file. We need it so
 * we'll have some place to specify the file operations
 * structure we want to use, and the function we use for
 * permissions. It's also possible to specify functions
 * to be called for anything else which could be done to
 * an inode (although we don't bother, we just put
 * NULL).
 */

static struct inode_operations Inode_Ops_File = {
        .permission = module_permission,        /* check for permissions */
};


/* check if unit is getting external power(AC) or no external power(DC) 
*  returns: 1 if docked ; returns 0 if on battery
*/
static int ds2786_isDocked(void)
{
	int ret=-1;
	ret=gpio_get_value(EXT_PWR_N);	//Read external power pin on the processor
	return (!ret);
}
/* enableLED(int sate) 
*  parameters: 1=set(illuminate); 0=off
*/
static void enableLed(int st)
{
	u8 ret=0;

	if (st && !g_led_state)
	{
		gpio_set_value(DOCK_CHG_INDICATOR, 1);
		g_led_state=1;
	}
	else if (!st && g_led_state) 
	{
		gpio_set_value(DOCK_CHG_INDICATOR, 0);
		g_led_state=0;
	}

	msleep(10);
}
static void blinkLed(void)
{
	enableLed(1);
	msleep(250);
	enableLed(0);
}

static inline int ds278x_read_reg(struct ds278x_info *info, int reg, u8 *val)
{
	int ret;

	ret = i2c_smbus_read_byte_data(info->client, reg);
	if (ret < 0) {
		//RJK, cmntd for Dennis dev_err(&info->client->dev, "register read failed\n");
		return ret;
	}

	*val = ret;
	return 0;
}

static inline int ds278x_read_reg16(struct ds278x_info *info, int reg_msb,
				    s16 *val)
{
	int ret;

	ret = swab16(i2c_smbus_read_word_data(info->client, reg_msb));
	if (ret < 0) {
		//RJK, cmntd for Dennis dev_err(&info->client->dev, "register read failed\n");
		return ret;
	}

	*val = ret;
	return 0;
}

u8 get_ds2786_capacity(void)
{
	return g_control->extCmd.BatteryLifePercent;
}
int get_ds2786_current(void)
{
	return g_control->extCmd.BatteryCurrent;
}
u16 get_ds2786_voltage(void)
{
	return g_control->extCmd.BatteryVoltage;
}

/* tst600- Temperature is being read from from gas Guage, could also be read from PIMC
* more accurate from pmic since  since signal TBAT goes to pmic and not to gas guage
*/
static int ds278x_get_temp(struct ds278x_info *info, int *temp)
{
	int err=0;
	s16 raw;

	mutex_lock(&battery_lock);
	err = ds278x_read_reg16(info, DS2786B_TEMP_MSB, &raw);
        mutex_unlock(&battery_lock);
        if (err){
                return err;
        }

	//*temp = ((raw/32)   *  125);    // degree mC
	*temp = ((raw/32) / 8);    // degree C
	//printk(KERN_ERR "raw=0x%x temp=0x%x\n", raw, *temp);

	return 0;
}

/* DS2786 reports current in signed units of 1.25ma, but the battery
 * class reports in units of µA,  */
static int ds2786_get_current(struct ds278x_info *info, int *current_uA)
{
	int err=0;
	s16 raw;

	mutex_lock(&battery_lock);
	err = ds278x_read_reg16(info, DS2786B_CURR_MSB, &raw);
	mutex_unlock(&battery_lock);
	if (err){
		return err;
	}

	*current_uA = (((raw / 16) * 1250) / info->rsns); // mA (20m Ohm current sense resistor in the hardware)
	//printk(KERN_ERR "current2: raw=0x%x, current_uA=0x%x \n", raw,*current_uA);
	return 0;
}

static int ds2786_get_voltage(struct ds278x_info *info, int *voltage_uA)
{
	s16 raw;
	int err=0;

	/*
	 * Voltage is measured in units of 1.22mV. The voltage is stored as
	 * a 10-bit number plus sign, in the upper bits of a 16-bit register
	 */
	mutex_lock(&battery_lock);
	err = ds278x_read_reg16(info, DS2786B_VOLT_MSB, &raw);
	mutex_unlock(&battery_lock);
	if (err){
             return err;
        }

	*voltage_uA = ((raw / 8) * 122)/100;   //RJK: WinCE calculation
	return 0;
}

static int ds2786_get_capacity(struct ds278x_info *info, u8 *capacity)
{
	int err=0;
	u8 raw;

	mutex_lock(&battery_lock);
	err = ds278x_read_reg(info, DS2786B_RELCAP, &raw);
	mutex_unlock(&battery_lock);
	if (err){
             return err;
        }
	/* Relative capacity is displayed with resolution 0.5 % */
	*capacity = raw/2 ;
	//printk(KERN_ERR "cpacity: raw=%d cap=%d\n", raw, *capacity);

	return 0;
}

static int ds2786_battery_read_status(struct ds278x_info *info)
{
	int err=0;
	int status;

	u8 LearnedCapacity, CellCapacity;
    	u32 mAhCapacity, mAh;

	err = info->ops->get_battery_voltage(info, &info->voltage_uV);
        if (err)
                return err;
	g_control->extCmd.BatteryVoltage=info->voltage_uV;

	err = info->ops->get_battery_current(info, &info->current_uA);
	if (err)
		return err;
	g_control->extCmd.BatteryCurrent=info->current_uA;

	err = info->ops->get_battery_capacity(info, &info->rem_capacity);
	if (err)
		return err;
	g_control->extCmd.BatteryLifePercent=info->rem_capacity;

	err = info->ops->get_battery_temp(info, &info->current_battery_temperature);
	if (err)
		return err;
	g_control->current_battery_temperature=info->current_battery_temperature;

	if (info->rem_capacity == 100) {
		status = POWER_SUPPLY_STATUS_FULL;
		g_control->extCmd.BatteryFlag=BATTERY_FLAG_HIGH;
	}
	else if (info->current_uA == 0) {
		status = POWER_SUPPLY_STATUS_NOT_CHARGING;
	}
	else if (info->current_uA < 0) {
		status = POWER_SUPPLY_STATUS_DISCHARGING;
	}
	else {
		status = POWER_SUPPLY_STATUS_CHARGING;
		g_control->extCmd.BatteryFlag=BATTERY_FLAG_CHARGING;
	}

	info->battery_status=status;
	g_control->battery_status=info->battery_status;
	//printk(KERN_ERR "battery_read_status: v=0x%x, cur=0x%x, cap=0x%x, stat=0x%x, temp=0x%x\n", 
	//		info->voltage_uV,info->current_uA,info->rem_capacity, info->battery_status, 
	//		info->current_battery_temperature);

	return 0;
}

/* RJK: This function is called by the power_supply module 
 * that reports to the power_supply subsystem.
 * this function is mainly called when power_supply_changed is 
 * called, and the values extracted
*/
static int ds278x_battery_get_property(struct power_supply *psy,
				       enum power_supply_property prop,
				       union power_supply_propval *val)
{
	struct ds278x_info *info = to_ds278x_info(psy);
	int ret=0;

	switch (prop) {
        case POWER_SUPPLY_PROP_STATUS:
                val->intval = info->battery_status;
                return 0;
        default:
                break;
        }

	ds2786_battery_read_status(info);

	switch (prop) {

	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = info->rem_capacity;
		break;

	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = info->voltage_uV;
		break;

	case POWER_SUPPLY_PROP_CURRENT_NOW:
		val->intval = info->current_uA;
		break;

	case POWER_SUPPLY_PROP_TEMP:
		val->intval = info->current_battery_temperature;
		break;

	default:
		ret = -EINVAL;
	}

	return ret;
}

static int ds278x_ac_get_property(struct power_supply *psy,
                enum power_supply_property psp,
                union power_supply_propval *val)
{
        struct ds278x_info *info = container_of(psy,
                        struct ds278x_info, ac);
	int ret=0;

        switch (psp) {
        case POWER_SUPPLY_PROP_ONLINE:
                val->intval = info->on_ac;
                break;
        default:
                break;
        }

        return 0;
}

static enum power_supply_property ds278x_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_TEMP,
};

static enum power_supply_property ds278x_ac_props[] = {
        POWER_SUPPLY_PROP_ONLINE,
};


static void ds278x_battery_init(struct ds278x_info *di)
{
	di->battery.type			= POWER_SUPPLY_TYPE_BATTERY;
	di->battery.properties			= ds278x_battery_props;
	di->battery.num_properties		= ARRAY_SIZE(ds278x_battery_props);
	di->battery.get_property		= ds278x_battery_get_property;
	//di->battery.external_power_changed	= NULL;
	di->battery.external_power_changed	= power_supply_changed;
}

static void ds278x_ac_init(struct ds278x_info *di)
{
        di->ac.name                        = "ac";
        di->ac.type                        = POWER_SUPPLY_TYPE_MAINS;
        di->ac.properties                  = ds278x_ac_props;
        di->ac.num_properties              = ARRAY_SIZE(ds278x_ac_props);
        di->ac.get_property                = ds278x_ac_get_property;
        //di->ac.external_power_changed    = NULL;
        di->ac.external_power_changed      = power_supply_changed;
}


static int ds278x_battery_remove(struct i2c_client *client)
{
	struct ds278x_info *info = i2c_get_clientdata(client);

	power_supply_unregister(&info->battery);
	kfree(info->battery.name);

	mutex_lock(&battery_lock);
	idr_remove(&battery_id, info->id);
	mutex_unlock(&battery_lock);

	kfree(info);
	return 0;
}

enum ds278x_num_id {
	DS2782 = 0,
	DS2786,
};

static struct ds278x_battery_ops ds278x_ops[] = {
	[DS2782] = {
//		.get_battery_current  = ds2782_get_current,
//		.get_battery_voltage  = ds2782_get_voltage,
//		.get_battery_capacity = ds2782_get_capacity,
	},
	[DS2786] = {
		.get_battery_current  = ds2786_get_current,
		.get_battery_voltage  = ds2786_get_voltage,
		.get_battery_capacity = ds2786_get_capacity,
		.get_battery_temp = ds278x_get_temp,
	}
};

static void display_battery_stats(struct ds278x_info *di) {
	int ret;
	union power_supply_propval value;

	/* read and print initial battery status */
	printk(KERN_ERR "on_AC=0x%x\n", di->on_ac);
	printk(KERN_ERR "ps_stat=0x%x\n", di->battery_status);
	printk(KERN_ERR "ps_cap=0x%x\n", di->rem_capacity);
	printk(KERN_ERR "ps_volt=0x%x\n", di->voltage_uV);
	printk(KERN_ERR "ps_cur=0x%x\n", di->current_uA);
	printk(KERN_ERR "ps_temp=0x%x\n", di->current_battery_temperature);
}

static void battery_params_init(void)
{
	g_control->battery_status=0;
	g_control->on_ac=-1;  //init to -1 so first update loop executes
	g_control->battery_level=0;
        g_control->power_led_enable=1;
        g_control->battery_critical_level_thresh=10;
        g_control->battery_warning_level_thresh=30;
        g_control->warning_led_flash_rate=2000;
        g_control->critical_led_flash_rate=1000;
        g_control->charging_led_flash_rate=3000;
}

static void ds2786_battery_update_status(struct ds278x_info *di)
{
        int retval;
	static int dispVal=0;
        int old_battery_status = di->battery_status;
        int old_docked_state = di->on_ac;
        int old_life_percent = di->rem_capacity;
	int docked_state=0;
	int docked_state_chgd=0;
	union power_supply_propval value;

#if 0 //RJK debug
//printk(KERN_ERR "RJK: ds2786_upd_stat\n");
if (!(dispVal%10)) {    // set to approximately 100 seconds
	display_battery_stats(di);
	dispVal=0;
}
dispVal++;
#endif
 	/* check if docked or not */	

 	docked_state=ds2786_isDocked();	
	g_control->on_ac = di->on_ac = docked_state;

	if (docked_state != old_docked_state){
		docked_state_chgd=1;
		printk(KERN_ERR "power_supply_changed: AC/DC \n");
               	power_supply_changed(&di->ac);
	}

	if (docked_state && docked_state_chgd)
	{
		printk(KERN_ERR "We are Docked!\n");
	}
	else if (!docked_state && docked_state_chgd)
	{
		printk(KERN_ERR "We are unDocked!\n");
	}
	
        if (di->battery_status == POWER_SUPPLY_STATUS_UNKNOWN)
           	g_control->full_counter = 0;

	retval=ds2786_battery_read_status(di);
	
	g_control->battery_level = di->rem_capacity;

	if (old_battery_status != POWER_SUPPLY_STATUS_UNKNOWN &&
             	 di->battery_status != old_battery_status) {
		/* pass supply change info to android */
	//printk(KERN_ERR "power_supply_changed: bs=%d : obs=%d\n", g_control->battery_status, old_battery_status);
               	power_supply_changed(&di->battery);
        }
	
        if (!g_control->on_ac) {
 		if(di->rem_capacity != old_life_percent) {
			/* pass supply change info to android */
	//printk(KERN_ERR "power_supply_changed: bl=%d : obl=%d\n", g_control->extCmd.BatteryLifePercent, old_life_percent);
                	power_supply_changed(&di->battery);
		}
   	}
}

static void ds2786_battery_work(struct work_struct *work)
{
        struct ds278x_info *di = container_of(work, struct ds278x_info, monitor_work.work);
        const int interval = HZ * 5; //check battery every 2 seconds 1HZ=1 second


        dev_dbg(di->dev, "%s\n", __func__);

        ds2786_battery_update_status(di);
	schedule_delayed_work(&di->monitor_work, interval);
}

static void ds2786_led_update(struct led_control *di)
{	

	if (ds2786_isDocked)	
	{
		if (g_control->power_led_enable) {
			enableLed(1);
		}
		else  {
			enableLed(0);
		}
	}
	else
	{
		if (g_control->battery_level <= g_control->battery_critical_level_thresh)
		{
			blinkLed();	
			di->interval=(g_control->critical_led_flash_rate/10); //Every 1/2 second
		}
		else if (g_control->battery_level <= g_control->battery_warning_level_thresh)
		{
			blinkLed();	
			di->interval=(g_control->warning_led_flash_rate/10);
		}
		else
		{
			/* enabling LED during battery does not make sense 
			* to me but need to expose to user anyway.
			*/
			 if (g_control->power_led_enable) {
               	        	 enableLed(1);
               		 }
               		 else  {
               		         enableLed(0);
                	}
		}
	}			
}

#if 0
/* RJK, not used, but leaving this in case we want to use LED to denote battery level in future.
 * This reason we pulled it is because the LED is not available when running on battery.
 */
static void ds2786_led_work(struct work_struct *work)
{
        struct led_control *di = container_of(work, struct led_control, led_work.work);
        di->interval = HZ * 1; //1 HZ = 1 second(1000)

        //dev_dbg(di->dev, "%s\n", __func__);

        ds2786_led_update(di);
	schedule_delayed_work(&di->led_work, di->interval);
}
#endif

static int ds278x_battery_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	struct ds278x_platform_data *pdata = client->dev.platform_data;
	struct ds278x_info *info;
	struct led_control *led;
	int ret;
	int num;
	union power_supply_propval value;

	printk(KERN_ERR " initializing ds278x_battery driver\n");

	/*
	 * ds2786 should have the sense resistor value set
	 * in the platform data
	 */
	if (id->driver_data == DS2786 && !pdata) {
		dev_err(&client->dev, "missing platform data for ds2786\n");
		return -EINVAL;
	}

	/* Get an ID for this battery */
	ret = idr_pre_get(&battery_id, GFP_KERNEL);
	if (ret == 0) {
		ret = -ENOMEM;
		goto fail_id;
	}

	mutex_lock(&battery_lock);
	ret = idr_get_new(&battery_id, client, &num);
	mutex_unlock(&battery_lock);
	if (ret < 0)
		goto fail_id;

	/* allocate memory for global control structure */
	g_control = kzalloc(sizeof(*g_control), GFP_KERNEL);
	if (!g_control) {
		ret = -ENOMEM;
		goto fail_info;
	}

	/* allocate memory for led control structure */
	led = kzalloc(sizeof(*led), GFP_KERNEL);
	if (!led) {
		ret = -ENOMEM;
		goto fail_info;
	}

	/* allocate memory for battery info structure */
	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info) {
		ret = -ENOMEM;
		goto fail_info;
	}

	info->battery.name = kasprintf(GFP_KERNEL, "%s-%d", client->name, num);
	if (!info->battery.name) {
		ret = -ENOMEM;
		goto fail_name;
	}

	if (id->driver_data == DS2786)
		info->rsns = pdata->rsns;

	i2c_set_clientdata(client, info);
	info->client = client;
	info->id = num;
	info->ops  = &ds278x_ops[id->driver_data];

	ds278x_battery_init(info);
	ds278x_ac_init(info);

//#if 0 //RJK
//RJK: should we reset device here?

	ret = power_supply_register(&client->dev, &info->battery);
	printk(KERN_INFO "RJK: registering battery power_supply\n");
	if (ret) {
		dev_err(&client->dev, "failed to register battery\n");
		goto fail_register;
	}
	ret = power_supply_register(&client->dev, &info->ac);
	printk(KERN_INFO "RJK: registering ac power_supply\n");
	if (ret) {
		dev_err(&client->dev, "failed to register battery\n");
		goto fail_register;
	}
//#endif

	/* Create procfs dir and files for HAL interface */

	PowerDir= proc_mkdir(POWER_PROCFS_DIRECTORY, NULL);
	if (PowerDir == NULL)
	{
		printk(KERN_ALERT "Error: Could not create /proc/%s directory\n",
                       POWER_PROCFS_DIRECTORY);
                return -ENOMEM;
	}
	else
	{
		/* create the /proc file for power status */
       		CmdFile = create_proc_entry(POWER_PROCFS_CMD_FILENAME, 0644, PowerDir);
       	 	if (CmdFile == NULL){
       	         	printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
       	                	POWER_PROCFS_CMD_FILENAME);
       	         	return -ENOMEM;
       	 	}
       	 	else
       	 	{
       	   		//CmdFile->owner = THIS_MODULE;
       	   		CmdFile->proc_iops = &Inode_Ops_File;
       	   		CmdFile->proc_fops = &File_Ops_Power_File;
       	   		CmdFile->mode = S_IFREG | S_IRUGO | S_IWUSR;
       	   		CmdFile->uid = 0;
       	   		CmdFile->gid = 0;
       	   		CmdFile->size = 80;
       		}

		/* create the /proc file for power status */
        	ExtCmdFile = create_proc_entry(POWER_PROCFS_EXTCMD_FILENAME, 0644, PowerDir);
        	if (ExtCmdFile == NULL){
               		 printk(KERN_ALERT "Error: Could not initialize /proc/%s\n",
               	       		 POWER_PROCFS_EXTCMD_FILENAME);
               	 	return -ENOMEM;
        	}
        	else
        	{
          		//ExtCmdFile->owner = THIS_MODULE;
          		ExtCmdFile->proc_iops = &Inode_Ops_File;
          		ExtCmdFile->proc_fops = &File_Ops_ExtPower_File;
          		ExtCmdFile->mode = S_IFREG | S_IRUGO | S_IWUSR;
          		ExtCmdFile->uid = 0;
          		ExtCmdFile->gid = 0;
          		ExtCmdFile->size = 80;
       		}
	}

	GasGaugeInitConfReg(info);
	GasGaugeInitEE(info);

	initDs2782ExtCmds();
	battery_params_init();

	info->on_ac=ds2786_isDocked();  //initialize AC or DC

	info->battery_status = POWER_SUPPLY_STATUS_UNKNOWN;
	g_control->battery_status = POWER_SUPPLY_STATUS_UNKNOWN;

	/* battery monitoring work */
	INIT_DELAYED_WORK_DEFERRABLE(&info->monitor_work, ds2786_battery_work);
	schedule_delayed_work(&info->monitor_work, HZ*40);

#if 0 /*LED is not available when device is not docked (use battery symbol to alert user) */
	/* LED action control */
	INIT_DELAYED_WORK(&led->led_work, ds2786_led_work);
	schedule_delayed_work(&led->led_work, HZ*100);
#endif

	msleep(100);
	goto success;

fail_workqueue:
fail_register:
	power_supply_unregister(&info->battery);
	kfree(info->battery.name);
fail_name:
	kfree(info);
fail_info:
	mutex_lock(&battery_lock);
	idr_remove(&battery_id, num);
	mutex_unlock(&battery_lock);
fail_id:
success:
	return ret;
}

static const struct i2c_device_id ds278x_id[] = {
	{"ds2782", DS2782},
	{"ds2786", DS2786},
	{},
};

static struct i2c_driver ds278x_battery_driver = {
	.driver 	= {
		.name	= "ds2786-battery",
	},
	.probe		= ds278x_battery_probe,
	.remove		= ds278x_battery_remove,
	.id_table	= ds278x_id,
};

static int __init ds278x_init(void)
{
if (gProductId == TST600)
	return i2c_add_driver(&ds278x_battery_driver);
}
module_init(ds278x_init);

static void __exit ds278x_exit(void)
{
	i2c_del_driver(&ds278x_battery_driver);
}
module_exit(ds278x_exit);

MODULE_AUTHOR("");
MODULE_DESCRIPTION("Fuel Gauge Driver");
